<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Impak_LC_REST {

    const ROUTE_NS = 'impak-lc/v1';

    public static function register_routes() {
        // Compare (legacy single endpoint) left for reference
        register_rest_route(
            self::ROUTE_NS,
            '/compare',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ __CLASS__, 'compare' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [
                        'doc_type' => [
                            'type' => 'string', 'required' => true,
                            'validate_callback' => function( $v ){ return is_string( $v ) && preg_match( '/^[a-zA-Z0-9_-]{1,20}$/', $v ); }
                        ],
                        'doc_number' => [
                            'type' => 'string', 'required' => true,
                            'validate_callback' => function( $v ){ return is_string( $v ) && preg_match( '/^[0-9a-zA-Z.-]{4,32}$/', $v ); }
                        ],
                        'amount' => [
                            'type' => 'integer', 'required' => true,
                            'validate_callback' => function( $v ){ return is_numeric( $v ) && $v > 0 && $v < 100000000; }
                        ],
                        'term' => [
                            'type' => 'integer', 'required' => true,
                            'validate_callback' => function( $v ){ return is_numeric( $v ) && $v > 0 && $v <= 120; }
                        ],
                    ],
                ],
            ]
        );

        // Multi-step routes
        register_rest_route(
            self::ROUTE_NS,
            '/register',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ __CLASS__, 'register_client' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [
                        'tipodocumento'   => [ 'type' => 'string',  'required' => true ],
                        'numerodocumento' => [ 'type' => 'string',  'required' => true ],
                        'montominimo'     => [ 'type' => 'integer', 'required' => true ],
                        'montomaximo'     => [ 'type' => 'integer', 'required' => true ],
                        'plazominimo'     => [ 'type' => 'integer', 'required' => true ],
                        'plazomaximo'     => [ 'type' => 'integer', 'required' => true ],
                    ],
                ],
            ]
        );

        register_rest_route(
            self::ROUTE_NS,
            '/process',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ __CLASS__, 'process_client' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [ 'procesoid' => [ 'type' => 'string', 'required' => true ] ],
                ],
            ]
        );

        register_rest_route(
            self::ROUTE_NS,
            '/complete',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ __CLASS__, 'complete_client' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [
                        'procesoid'          => [ 'type' => 'string', 'required' => true ],
                        'nombres'            => [ 'type' => 'string', 'required' => true ],
                        'apellidos'          => [ 'type' => 'string', 'required' => true ],
                        'correoelectronico'  => [ 'type' => 'string', 'required' => true ],
                        'telefono'           => [ 'type' => 'string', 'required' => true ],
                    ],
                ],
            ]
        );

        register_rest_route(
            self::ROUTE_NS,
            '/results',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [ __CLASS__, 'get_results' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [ 'procesoid' => [ 'type' => 'string', 'required' => true ] ],
                ],
            ]
        );

        register_rest_route(
            self::ROUTE_NS,
            '/notify',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [ __CLASS__, 'notify_selection' ],
                    'permission_callback' => [ __CLASS__, 'permissions' ],
                    'args'                => [
                        'procesoid' => [ 'type' => 'string', 'required' => true ],
                        'entidadid' => [ 'type' => 'string', 'required' => true ],
                    ],
                ],
            ]
        );
    }

    public static function permissions( $request ) {
        // Accept either WP REST nonce (for logged-in) or our public nonce for anonymous visitors
        $wp_rest = $request->get_header( 'x-wp-nonce' );
        $impak   = $request->get_header( 'x-impak-nonce' );
        if ( $wp_rest && wp_verify_nonce( $wp_rest, 'wp_rest' ) ) {
            return true;
        }
        if ( $impak && wp_verify_nonce( $impak, 'impak_lc' ) ) {
            return true;
        }
        return false;
    }

    protected static function rate_limit_key( $bucket = 'global' ) {
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : 'unknown';
        return 'impak_lc_rl_' . md5( $bucket . '|' . $ip );
    }

    protected static function check_rate_limit( $bucket = 'global', $ttl = 1 ) {
        // Per-endpoint simple rate limit: default 1 request / 1 second per IP per bucket
        $key = self::rate_limit_key( $bucket );
        $last = get_transient( $key );
        if ( $last ) { return false; }
        set_transient( $key, time(), max( 1, (int) $ttl ) );
        return true;
    }

    protected static function build_url( $path, $replacements = [] ) {
        $settings   = get_option( Impak_LC_Settings::OPTION_NAME, [] );
        $api_base   = isset( $settings['api_base'] ) ? trailingslashit( $settings['api_base'] ) : '';
        if ( empty( $api_base ) || empty( $path ) ) { return ''; }
        foreach ( $replacements as $key => $value ) {
            $path = str_replace( '{' . $key . '}', rawurlencode( $value ), $path );
        }
        return $api_base . ltrim( $path, '/' );
    }

    protected static function headers() {
        $settings   = get_option( Impak_LC_Settings::OPTION_NAME, [] );
        $api_key     = isset( $settings['api_key'] ) ? $settings['api_key'] : '';
        $auth_header = ! empty( $settings['auth_header'] ) ? $settings['auth_header'] : 'Authorization';
        $auth_scheme = ! empty( $settings['auth_scheme'] ) ? $settings['auth_scheme'] : 'Bearer';
        $headers = [ 'Accept' => 'application/json' ];
        if ( $api_key ) { $headers[ $auth_header ] = trim( $auth_scheme . ' ' . $api_key ); }
        return $headers;
    }

    protected static function post_json( $url, $payload, $extra_headers = [] ) {
        $args = [
            'method'  => 'POST',
            'headers' => array_merge( self::headers(), [ 'Content-Type' => 'application/json' ], $extra_headers ),
            'body'    => wp_json_encode( $payload ),
            'timeout' => 30,
        ];
        return wp_remote_request( $url, $args );
    }

    protected static function get( $url, $extra_headers = [] ) {
        $args = [
            'method'  => 'GET',
            'headers' => array_merge( self::headers(), $extra_headers ),
            'timeout' => 30,
        ];
        return wp_remote_request( $url, $args );
    }

    public static function compare( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'compare', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }

        $settings = get_option( Impak_LC_Settings::OPTION_NAME, [] );
        $api_base    = isset( $settings['api_base'] ) ? trailingslashit( $settings['api_base'] ) : '';
        $api_endpoint= isset( $settings['api_endpoint'] ) ? ltrim( $settings['api_endpoint'], '/' ) : '';
        $api_key     = isset( $settings['api_key'] ) ? $settings['api_key'] : '';
        $auth_header = ! empty( $settings['auth_header'] ) ? $settings['auth_header'] : 'Authorization';
        $auth_scheme = ! empty( $settings['auth_scheme'] ) ? $settings['auth_scheme'] : 'Bearer';

        if ( empty( $api_base ) || empty( $api_endpoint ) || empty( $api_key ) ) {
            return new WP_Error( 'config_missing', __( 'Falta configuración de API en ajustes del plugin.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        }

        $payload = [
            'doc_type'   => $request->get_param( 'doc_type' ),
            'doc_number' => $request->get_param( 'doc_number' ),
            'amount'     => (int) $request->get_param( 'amount' ),
            'term'       => (int) $request->get_param( 'term' ),
        ];

        $url = $api_base . $api_endpoint;
        $args = [
            'method'      => 'POST',
            'headers'     => [
                $auth_header => trim( $auth_scheme . ' ' . $api_key ),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body'        => wp_json_encode( $payload ),
            'timeout'     => 20,
            'data_format' => 'body',
        ];

        $response = wp_remote_request( $url, $args );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'api_error', __( 'No se pudo conectar con el servicio externo.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $response->get_error_message() ] );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $data = [ 'raw' => $body ];
        }

        // Map non-2xx to WP error
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'api_http_' . $code, __( 'Error de la API externa.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $data ] );
        }

        // Success
        return new WP_REST_Response( [ 'ok' => true, 'data' => $data ], 200 );
    }

    public static function register_client( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'register', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }
        $path = Impak_LC_Settings::get( 'register_path' );
        if ( ! $path ) return new WP_Error( 'config_missing', __( 'Falta ruta de Registro.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        $url = self::build_url( $path );
        $payload = [
            'tipodocumento'   => $request->get_param( 'tipodocumento' ),
            'numerodocumento' => $request->get_param( 'numerodocumento' ),
            'montominimo'     => (int) $request->get_param( 'montominimo' ),
            'montomaximo'     => (int) $request->get_param( 'montomaximo' ),
            'plazominimo'     => (int) $request->get_param( 'plazominimo' ),
            'plazomaximo'     => (int) $request->get_param( 'plazomaximo' ),
        ];
        $response = self::post_json( $url, $payload );
        return self::handle_response( $response );
    }

    public static function process_client( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'process', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }
        $path = Impak_LC_Settings::get( 'process_path' );
        if ( ! $path ) return new WP_Error( 'config_missing', __( 'Falta ruta de Procesar.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        $url  = self::build_url( $path, [ 'procesoid' => $request->get_param( 'procesoid' ) ] );
        // Primary: GET as per provider docs. Retry on transient upstream failures (5xx) up to 3 attempts
        $attempts = 0;
        $response = null;
        $last_code = null;
        do {
            $attempts++;
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( sprintf( '[Impak LC] process GET attempt %d URL: %s', $attempts, $url ) );
            }
            $response = self::get( $url );
            if ( is_wp_error( $response ) ) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( '[Impak LC] process GET wp_error: ' . $response->get_error_message() );
                }
                // wp_error: treat as transient and retry
                if ( $attempts < 3 ) { sleep(1); }
                continue;
            }
            $code = wp_remote_retrieve_response_code( $response );
            $last_code = $code;
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( sprintf( '[Impak LC] process GET http %d body: %s', $code, wp_remote_retrieve_body( $response ) ) );
            }
            if ( $code >= 500 && $code < 600 && $attempts < 3 ) {
                sleep(1);
            } else {
                break;
            }
        } while ( $attempts < 3 );

        // Fallback: some implementations expect POST -> retry once with POST on 405
        if ( ! is_wp_error( $response ) && (int) $last_code === 405 ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[Impak LC] process fallback to POST: ' . $url );
            }
            $response = self::post_json( $url, (object)[] );
        }

        return self::handle_response( $response );
    }

    public static function complete_client( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'complete', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }
        $path = Impak_LC_Settings::get( 'complete_path' );
        if ( ! $path ) return new WP_Error( 'config_missing', __( 'Falta ruta de Completar.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        $url  = self::build_url( $path, [ 'procesoid' => $request->get_param( 'procesoid' ) ] );
        $payload = [
            'nombres'           => $request->get_param( 'nombres' ),
            'apellidos'         => $request->get_param( 'apellidos' ),
            'correoelectronico' => $request->get_param( 'correoelectronico' ),
            'telefono'          => $request->get_param( 'telefono' ),
        ];
        $response = self::post_json( $url, $payload );
        return self::handle_response( $response );
    }

    public static function get_results( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'results', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }
        $path = Impak_LC_Settings::get( 'results_path' );
        if ( ! $path ) return new WP_Error( 'config_missing', __( 'Falta ruta de Resultados.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        $url  = self::build_url( $path, [ 'procesoid' => $request->get_param( 'procesoid' ) ] );
        $response = self::get( $url );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'api_error', __( 'No se pudo conectar con el servicio externo.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $response->get_error_message() ] );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) { $data = [ 'raw' => $body ]; }
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'api_http_' . $code, __( 'Error de la API externa.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $data ] );
        }
        $normalized = self::normalize_results( $data );
        return new WP_REST_Response( [ 'ok' => true, 'data' => $normalized ], 200 );
    }

    public static function notify_selection( WP_REST_Request $request ) {
        if ( ! self::check_rate_limit( 'notify', 1 ) ) {
            return new WP_Error( 'too_many', __( 'Demasiadas solicitudes, intenta de nuevo en unos segundos.', 'impak-loan-comparator' ), [ 'status' => 429 ] );
        }
        $path = Impak_LC_Settings::get( 'notify_path' );
        if ( ! $path ) return new WP_Error( 'config_missing', __( 'Falta ruta de Notificar.', 'impak-loan-comparator' ), [ 'status' => 500 ] );
        $url  = self::build_url( $path, [
            'procesoid' => $request->get_param( 'procesoid' ),
            'entidadid' => $request->get_param( 'entidadid' ),
        ] );
        $response = self::get( $url );
        return self::handle_response( $response );
    }

    protected static function handle_response( $response ) {
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'api_error', __( 'No se pudo conectar con el servicio externo.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $response->get_error_message() ] );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $data = [ 'raw' => $body ];
        }
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'api_http_' . $code, __( 'Error de la API externa.', 'impak-loan-comparator' ), [ 'status' => 502, 'details' => $data ] );
        }
        return new WP_REST_Response( [ 'ok' => true, 'data' => $data ], 200 );
    }

    protected static function normalize_results( $data ) {
        // Expect a list in common locations
        $list = [];
        if ( is_array( $data ) ) {
            if ( isset( $data['resultados'] ) && is_array( $data['resultados'] ) ) {
                $list = $data['resultados'];
            } elseif ( isset( $data['results'] ) && is_array( $data['results'] ) ) {
                $list = $data['results'];
            } elseif ( isset( $data['data'] ) && is_array( $data['data'] ) && isset( $data['data']['resultados'] ) && is_array( $data['data']['resultados'] ) ) {
                $list = $data['data']['resultados'];
            } elseif ( array_keys( $data ) === range( 0, count( $data ) - 1 ) ) {
                // Root-level array
                $list = $data;
            }
        }

        $out = [];
        foreach ( $list as $item ) {
            if ( ! is_array( $item ) ) { continue; }
            $out[] = [
                'id'        => isset( $item['id'] ) ? $item['id'] : ( isset( $item['entidadid'] ) ? $item['entidadid'] : '' ),
                'nombre'    => isset( $item['nombre'] ) ? $item['nombre'] : ( isset( $item['entidad'] ) ? $item['entidad'] : '' ),
                'codigo'    => isset( $item['codigo'] ) ? $item['codigo'] : ( isset( $item['code'] ) ? $item['code'] : '' ),
                'codigoentidad' => isset( $item['codigoentidad'] ) ? $item['codigoentidad'] : ( isset( $item['codigo'] ) ? $item['codigo'] : '' ),
                'prioridad' => isset( $item['prioridad'] ) ? $item['prioridad'] : ( isset( $item['ranking'] ) ? $item['ranking'] : null ),
                'monto'     => isset( $item['monto'] ) ? $item['monto'] : ( isset( $item['montototal'] ) ? $item['montototal'] : null ),
                'plazo'     => isset( $item['plazo'] ) ? $item['plazo'] : ( isset( $item['dias'] ) ? $item['dias'] : null ),
                'tasa'      => isset( $item['tasa'] ) ? $item['tasa'] : ( isset( $item['tcea'] ) ? $item['tcea'] : null ),
                'url'       => isset( $item['url'] ) ? $item['url'] : ( isset( $item['link'] ) ? $item['link'] : '' ),
            ];
        }

        return [ 'resultados' => $out ];
    }
}
