<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Impak_LC_Settings {

    const OPTION_GROUP = 'impak_lc_settings_group';
    const OPTION_NAME  = 'impak_lc_settings';

    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'add_menu' ] );
        add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
    }

    public static function add_menu() {
        add_options_page(
            __( 'Impak Loan Comparator', 'impak-loan-comparator' ),
            __( 'Impak Loan Comparator', 'impak-loan-comparator' ),
            'manage_options',
            'impak-lc-settings',
            [ __CLASS__, 'render_page' ]
        );
    }

    public static function register_settings() {
        register_setting( self::OPTION_GROUP, self::OPTION_NAME, [ __CLASS__, 'sanitize' ] );

        add_settings_section(
            'impak_lc_section_api',
            __( 'Configuración de API', 'impak-loan-comparator' ),
            '__return_false',
            'impak-lc-settings'
        );

        add_settings_field(
            'api_base',
            __( 'API Base URL', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_api',
            [ 'id' => 'api_base', 'placeholder' => 'https://api.example.com' ]
        );

        add_settings_field(
            'api_endpoint',
            __( 'Endpoint de comparación', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_api',
            [ 'id' => 'api_endpoint', 'placeholder' => '/compare' ]
        );

        add_settings_field(
            'api_key',
            __( 'API Key (se guarda cifrada si es posible)', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_password' ],
            'impak-lc-settings',
            'impak_lc_section_api',
            [ 'id' => 'api_key' ]
        );

        add_settings_field(
            'auth_header',
            __( 'Encabezado de autenticación', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_api',
            [ 'id' => 'auth_header', 'placeholder' => 'Authorization' ]
        );

        add_settings_field(
            'auth_scheme',
            __( 'Esquema (p.ej. Bearer)', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_api',
            [ 'id' => 'auth_scheme', 'placeholder' => 'Bearer' ]
        );

        // Paths
        add_settings_section(
            'impak_lc_section_paths',
            __( 'Rutas de la API externa', 'impak-loan-comparator' ),
            function(){ echo '<p>' . esc_html__( 'Usa {procesoid} y {entidadid} como placeholders donde corresponda.', 'impak-loan-comparator' ) . '</p>'; },
            'impak-lc-settings'
        );

        $paths = [
            'register_path' => [ __( 'Registrar', 'impak-loan-comparator' ), 'registro/registrar' ],
            'process_path'  => [ __( 'Procesar', 'impak-loan-comparator' ), 'registro/procesar/{procesoid}' ],
            'complete_path' => [ __( 'Completar', 'impak-loan-comparator' ), 'registro/completar/{procesoid}' ],
            'results_path'  => [ __( 'Resultados', 'impak-loan-comparator' ), 'resumen/{procesoid}' ],
            'notify_path'   => [ __( 'Notificar', 'impak-loan-comparator' ), 'resultados/notificar/{procesoid}/{entidadid}' ],
        ];
        foreach ( $paths as $id => $meta ) {
            add_settings_field(
                $id,
                $meta[0],
                [ __CLASS__, 'field_text' ],
                'impak-lc-settings',
                'impak_lc_section_paths',
                [ 'id' => $id, 'placeholder' => $meta[1] ]
            );
        }

        // Shortcode helper (display only)
        add_settings_section(
            'impak_lc_section_shortcode',
            __( 'Shortcode', 'impak-loan-comparator' ),
            '__return_false',
            'impak-lc-settings'
        );

        add_settings_field(
            'shortcode_display',
            __( 'Usa este shortcode', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_shortcode_display' ],
            'impak-lc-settings',
            'impak_lc_section_shortcode',
            []
        );

        // Form options (amounts and terms)
        add_settings_section(
            'impak_lc_section_form_options',
            __( 'Opciones del formulario', 'impak-loan-comparator' ),
            function(){ echo '<p>' . esc_html__( 'Configura las opciones de los selects. Una opción por línea. Formato: valor|etiqueta. Si no incluyes etiqueta, se usará el mismo valor como etiqueta.', 'impak-loan-comparator' ) . '</p>'; },
            'impak-lc-settings'
        );

        add_settings_field(
            'amount_options',
            __( 'Opciones de monto', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_textarea' ],
            'impak-lc-settings',
            'impak_lc_section_form_options',
            [
                'id' => 'amount_options',
                'placeholder' => "50-200|50 - 200\n200-500|200 - 500\n500-1000|500 - 1000\n1000-2000|1000 - 2000\n2000-5000|2000 - 5000"
            ]
        );

        add_settings_field(
            'term_options',
            __( 'Opciones de plazo (días)', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_textarea' ],
            'impak-lc-settings',
            'impak_lc_section_form_options',
            [
                'id' => 'term_options',
                'placeholder' => "30\n60\n90"
            ]
        );

        add_settings_field(
            'amount_placeholder',
            __( 'Placeholder de monto', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_form_options',
            [ 'id' => 'amount_placeholder', 'placeholder' => __( 'Selecciona el monto', 'impak-loan-comparator' ) ]
        );

        add_settings_field(
            'term_placeholder',
            __( 'Placeholder de plazo', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_form_options',
            [ 'id' => 'term_placeholder', 'placeholder' => __( 'Selecciona el plazo', 'impak-loan-comparator' ) ]
        );

        // New: Amount max for slider
        add_settings_field(
            'amount_max',
            __( 'Monto máximo (slider)', 'impak-loan-comparator' ),
            [ __CLASS__, 'field_text' ],
            'impak-lc-settings',
            'impak_lc_section_form_options',
            [ 'id' => 'amount_max', 'placeholder' => '1500' ]
        );
    }

    public static function sanitize( $input ) {
        $out = [];
        $out['api_base']    = isset( $input['api_base'] ) ? esc_url_raw( trim( $input['api_base'] ) ) : '';
        $out['api_endpoint']= isset( $input['api_endpoint'] ) ? sanitize_text_field( trim( $input['api_endpoint'] ) ) : '';
        $out['auth_header'] = isset( $input['auth_header'] ) ? sanitize_text_field( trim( $input['auth_header'] ) ) : 'Authorization';
        $out['auth_scheme'] = isset( $input['auth_scheme'] ) ? sanitize_text_field( trim( $input['auth_scheme'] ) ) : 'Bearer';

        // Paths sanitize
        foreach ( [ 'register_path','process_path','complete_path','results_path','notify_path' ] as $k ) {
            $out[ $k ] = isset( $input[ $k ] ) ? ltrim( sanitize_text_field( $input[ $k ] ), '/' ) : '';
        }

        // Keep previous key if empty in submission
        $opts = get_option( self::OPTION_NAME, [] );
        if ( ! empty( $input['api_key'] ) ) {
            $out['api_key'] = trim( $input['api_key'] );
        } else {
            $out['api_key'] = isset( $opts['api_key'] ) ? $opts['api_key'] : '';
        }

        // Store raw textarea for options (parsed later at render time)
        $out['amount_options'] = isset( $input['amount_options'] ) ? implode( "\n", array_map( 'sanitize_text_field', array_filter( array_map( 'trim', explode( "\n", (string) $input['amount_options'] ) ) ) ) ) : ( isset( $opts['amount_options'] ) ? $opts['amount_options'] : '' );
        $out['term_options']   = isset( $input['term_options'] ) ? implode( "\n", array_map( 'sanitize_text_field', array_filter( array_map( 'trim', explode( "\n", (string) $input['term_options'] ) ) ) ) ) : ( isset( $opts['term_options'] ) ? $opts['term_options'] : '' );
        $out['amount_placeholder'] = isset( $input['amount_placeholder'] ) ? sanitize_text_field( $input['amount_placeholder'] ) : ( isset( $opts['amount_placeholder'] ) ? $opts['amount_placeholder'] : '' );
        $out['term_placeholder']   = isset( $input['term_placeholder'] ) ? sanitize_text_field( $input['term_placeholder'] ) : ( isset( $opts['term_placeholder'] ) ? $opts['term_placeholder'] : '' );
        // New sanitize for amount_max
        $prev_amount_max = isset( $opts['amount_max'] ) ? (int) $opts['amount_max'] : 1500;
        if ( isset( $input['amount_max'] ) && $input['amount_max'] !== '' ) {
            $val = (int) $input['amount_max'];
            if ( $val < 50 ) { $val = 50; }
            if ( $val > 50000 ) { $val = 50000; }
            $out['amount_max'] = $val;
        } else {
            $out['amount_max'] = $prev_amount_max;
        }
        return $out;
    }

    public static function get( $key, $default = '' ) {
        $opts = get_option( self::OPTION_NAME, [] );
        return isset( $opts[ $key ] ) ? $opts[ $key ] : $default;
    }

    public static function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Impak Loan Comparator - Ajustes', 'impak-loan-comparator' ); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields( self::OPTION_GROUP ); ?>
                <?php do_settings_sections( 'impak-lc-settings' ); ?>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public static function field_text( $args ) {
        $id = $args['id'];
        $val = esc_attr( self::get( $id ) );
        $ph  = isset( $args['placeholder'] ) ? esc_attr( $args['placeholder'] ) : '';
        echo '<input type="text" class="regular-text" id="' . esc_attr( $id ) . '" name="' . self::OPTION_NAME . '[' . esc_attr( $id ) . ']" value="' . $val . '" placeholder="' . $ph . '" />';
    }

    public static function field_password( $args ) {
        $id = $args['id'];
        $val = esc_attr( self::get( $id ) );
        echo '<input type="password" class="regular-text" id="' . esc_attr( $id ) . '" name="' . self::OPTION_NAME . '[' . esc_attr( $id ) . ']" value="' . $val . '" autocomplete="off" />';
    }

    public static function field_shortcode_display() {
        $code = '[impak_loan_comparator]';
        echo '<input type="text" class="regular-text" readonly value="' . esc_attr( $code ) . '" onfocus="this.select();" />';
        echo '<p class="description">' . esc_html__( 'Copia y pega este shortcode en una entrada o página para mostrar el comparador.', 'impak-loan-comparator' ) . '</p>';
    }

    public static function field_textarea( $args ) {
        $id = $args['id'];
        $val = self::get( $id );
        $ph  = isset( $args['placeholder'] ) ? $args['placeholder'] : '';
        echo '<textarea class="large-text code" rows="6" id="' . esc_attr( $id ) . '" name="' . self::OPTION_NAME . '[' . esc_attr( $id ) . ']" placeholder="' . esc_attr( $ph ) . '">';
        echo esc_textarea( $val );
        echo '</textarea>';
        echo '<p class="description">' . esc_html__( 'Una opción por línea. Usa "valor|etiqueta". Si no incluyes etiqueta, se mostrará el mismo valor.', 'impak-loan-comparator' ) . '</p>';
    }
}
